/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  AbsoluteTime,
  AmountJson,
  MerchantContractChoice,
  MerchantInfo,
  PaymentInsufficientBalanceDetails,
  TransactionIdStr
} from "@gnu-taler/taler-util";
import { ErrorAlertView } from "../../components/CurrentAlerts.js";
import { Loading } from "../../components/Loading.js";
import { ErrorAlert } from "../../context/alert.js";
import { ButtonHandler } from "../../mui/handlers.js";
import { compose, StateViewMap } from "../../utils/index.js";
import { useComponentState } from "./state.js";
import { BaseView, PaymentStates } from "./views.js";

export interface Props {
  talerPayUri: string;
  goToWalletManualWithdraw: (amount?: string) => Promise<void>;
  cancel: () => Promise<void>;
  onSuccess: (tx: string) => Promise<void>;
}

export type State =
  | State.Loading
  | State.LoadingUriError
  | PaymentStates;

export namespace State {
  
  export interface Loading {
    status: "loading";
    error: undefined;
  }

  export interface LoadingUriError {
    status: "error";
    error: ErrorAlert;
  }

  interface BaseInfo {
    amount: AmountJson;
    summary: string;
    minimum_age: number | undefined,
    expiration: AbsoluteTime;
    merchant: MerchantInfo | undefined;
    transactionId: TransactionIdStr;
    shareUri: string;
    error: undefined;
    goToWalletManualWithdraw: (amount?: string) => Promise<void>;
    cancel: () => Promise<void>;
  }

  export interface NoEnoughBalance extends BaseInfo {
    status: "no-enough-balance";
    balanceDetails: PaymentInsufficientBalanceDetails | undefined;
    effective: undefined;
    choices: undefined | {
      list: MerchantContractChoice[]
      index: number;
      select: (d: number) => void;
    },
  }

  export interface Ready extends BaseInfo {
    status: "ready";
    payHandler: ButtonHandler;
    effective: AmountJson;
    choices: undefined | {
      list: MerchantContractChoice[]
      index: number;
      select: (d: number) => void;
    },
  }

  export interface Confirmed extends BaseInfo {
    status: "confirmed";
    effective: undefined;
    message?: string;
    paid: boolean;
  }

}

const viewMapping: StateViewMap<State> = {
  loading: Loading,
  error: ErrorAlertView,
  "no-enough-balance": BaseView,
  confirmed: BaseView,
  ready: BaseView,
};

export const PaymentPage = compose(
  "Payment",
  (p: Props) => useComponentState(p),
  viewMapping,
);
